//*****************************************************************************
//
//! @file hello_world.c
//!
//! @brief A simple "Hello World" example.
//!
//! Purpose: This example prints a "Hello World" message with some device info.
//!
//! Printing takes place over the ITM at 1M Baud.
//!
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2019, Ambiq Micro
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
// 
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision SBL3P_v0_dryrun-132-g52879a9c9 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include "am_mcu_apollo.h"
#include "am_bsp.h"
#include "am_devices_mspi_psram_aps6404l.h"
#include "am_devices_mspi_rm69330.h"
#include "am_util.h"
#include "am_util_stdio.h"
#include "am_util_delay.h"
#include <string.h>
#include "demo09_350_350_24.h"
#include "demo30_350_350_24.h"

#define MSPI_PSRAM_MODULE       0
#define MSPI_TEST_MODULE        1

uint32_t  DMATCBBuffer[2560];

#define FB_SIZE (3*340*340)

uint8_t   g_FB[FB_SIZE] = {0x0};

volatile uint8_t g_TE_complete = 0;
#define AM_BSP_GPIO_DISPLAY_HBOARD_TE (48)
const am_hal_gpio_pincfg_t g_TE_config =
{
    .uFuncSel = 3,
    .eIntDir = AM_HAL_GPIO_PIN_INTDIR_HI2LO,
    .eGPInput = AM_HAL_GPIO_PIN_INPUT_ENABLE,
    .eDriveStrength  = AM_HAL_GPIO_PIN_DRIVESTRENGTH_2MA,
};
static void teInt_handler(void)
{
    g_TE_complete++;
}

am_devices_mspi_psram_config_t MSPI_PSRAM_QuadCE0MSPIConfig =
{
    .eDeviceConfig            = AM_HAL_MSPI_FLASH_QUAD_CE0,
    .eClockFreq               = AM_HAL_MSPI_CLK_48MHZ,
    .eMixedMode               = AM_HAL_MSPI_XIPMIXED_NORMAL,
    .ui32NBTxnBufLength       = sizeof(DMATCBBuffer) / sizeof(uint32_t),
    .pNBTxnBuf                = DMATCBBuffer,
    .ui32ScramblingStartAddr  = 0,
    .ui32ScramblingEndAddr    = 0,
};

void            *g_MSPIPsramHandle;
void            *g_PsramHandle;

void            *g_MSPIDisplayHandle;
void            *g_DisplayHandle;
//! MSPI interrupts.
static const IRQn_Type mspi_psram_interrupts[] =
{
    MSPI0_IRQn,
#if defined(AM_PART_APOLLO3P)
    MSPI1_IRQn,
    MSPI2_IRQn,
#endif
};

//*****************************************************************************
//
// MSPI ISRs.
//
//*****************************************************************************
void am_mspi0_isr(void)
{
    uint32_t      ui32Status;

    am_hal_mspi_interrupt_status_get(g_MSPIPsramHandle, &ui32Status, false);

    am_hal_mspi_interrupt_clear(g_MSPIPsramHandle, ui32Status);

    am_hal_mspi_interrupt_service(g_MSPIPsramHandle, ui32Status);
}
#if defined(AM_PART_APOLLO3P)
void am_mspi1_isr(void)
{
    uint32_t      ui32Status;

    am_hal_mspi_interrupt_status_get(g_MSPIDisplayHandle, &ui32Status, false);

    am_hal_mspi_interrupt_clear(g_MSPIDisplayHandle, ui32Status);

    am_hal_mspi_interrupt_service(g_MSPIDisplayHandle, ui32Status);
}
#endif

//*****************************************************************************
//
// Interrupt handler for the GPIO pins.
//
//*****************************************************************************
void am_gpio_isr(void)
{
    AM_HAL_GPIO_MASKCREATE(GpioIntMask);
 
    am_hal_gpio_interrupt_status_get(false, pGpioIntMask);

    if ((GpioIntMask.U.Msk[1] >> AM_BSP_GPIO_DISPLAY_HBOARD_TE-32) & 1 == 1){
        am_hal_gpio_interrupt_clear(AM_HAL_GPIO_MASKBIT(pGpioIntMask, AM_BSP_GPIO_DISPLAY_HBOARD_TE));
        am_hal_gpio_interrupt_service(pGpioIntMask);
    } 	
}

void am_devices_mspi_display_gpio_init(void)
{
    am_hal_gpio_pinconfig(13, g_AM_HAL_GPIO_OUTPUT);//MCU_CTL_CHGR_EN1 for 500mA
    am_hal_gpio_output_set(13);
    
    am_hal_gpio_pinconfig(3, g_AM_HAL_GPIO_OUTPUT);//5V output en
    am_hal_gpio_output_set(3);
    am_hal_gpio_pinconfig(32, g_AM_HAL_GPIO_OUTPUT);//2.8V output en
    am_hal_gpio_output_set(32);
    
    am_hal_gpio_pinconfig(47, g_AM_HAL_GPIO_OUTPUT); // AM_BSP_GPIO_DISPLAY_RES
    am_hal_gpio_output_clear(47);

    am_hal_gpio_pinconfig(48, g_TE_config); //AM_BSP_GPIO_DISPLAY_TE

    am_hal_gpio_pinconfig(46, g_AM_HAL_GPIO_INPUT);//AM_BSP_GPIO_DISPLAY_EN

    am_hal_gpio_pinconfig(56, g_AM_HAL_GPIO_OUTPUT);//AM_BSP_GPIO_DISPLAY_TPS65136_EN
    am_hal_gpio_output_clear(56);

    am_hal_gpio_pinconfig(55, g_AM_HAL_GPIO_OUTPUT);//AM_BSP_GPIO_DISPLAY_1V8_EN
    am_hal_gpio_output_set(55);

    am_hal_gpio_pinconfig(49, g_AM_HAL_GPIO_OUTPUT);//AM_BSP_GPIO_DISPLAY_3V3_EN
    am_hal_gpio_output_set(49);
}

uint32_t
mspi_psram_init(void)
{
    uint32_t ui32Status;
    ui32Status = am_devices_mspi_psram_init(MSPI_PSRAM_MODULE,
                                            &MSPI_PSRAM_QuadCE0MSPIConfig,
                                            &g_PsramHandle,
                                            &g_MSPIPsramHandle);
    NVIC_EnableIRQ(mspi_psram_interrupts[MSPI_PSRAM_MODULE]);
    am_hal_interrupt_master_enable();

    return ui32Status;
}

int
init_mspi_psram_data(void)
{
    uint32_t      ui32Status;

    am_util_stdio_printf("Writing a known pattern to psram!\n");

    ui32Status = am_devices_mspi_psram_write(g_PsramHandle, (uint8_t *)&gImage_demo09_340_340_24[0], 0, FB_SIZE, true);
    if (AM_DEVICES_MSPI_PSRAM_STATUS_SUCCESS != ui32Status)
    {
        am_util_stdio_printf("Failed to write buffer 1 to PSRAM Device!\n");
        return -1;
    }

    ui32Status = am_devices_mspi_psram_read(g_PsramHandle, (uint8_t *)&g_FB[0], 0, FB_SIZE, true);
    if (AM_DEVICES_MSPI_PSRAM_STATUS_SUCCESS != ui32Status)
    {
        am_util_stdio_printf("Failed to read buffer 1 to PSRAM Device!\n");
        return -1;
    }

    ui32Status = am_devices_mspi_psram_write(g_PsramHandle, (uint8_t *)&gImage_demo30_340_340_24[0], FB_SIZE, FB_SIZE, true);
    if (AM_DEVICES_MSPI_PSRAM_STATUS_SUCCESS != ui32Status)
    {
       am_util_stdio_printf("Failed to write buffer 2 to PSRAM Device!\n");
       return -1;
    }
    return 0;
}

static uint32_t  g_MspiDisplayQBuffer[(AM_HAL_MSPI_CQ_ENTRY_SIZE / 4) * (2 + 1)];
static am_devices_mspi_rm69330_config_t QuadDisplayMSPICfg =
{
    .eDeviceConfig            = AM_HAL_MSPI_FLASH_QUAD_CE0,
    .eClockFreq               = AM_HAL_MSPI_CLK_48MHZ,
    .eMixedMode               = AM_HAL_MSPI_XIPMIXED_AD4,
    .ui32NBTxnBufLength       = sizeof(g_MspiDisplayQBuffer) / sizeof(uint32_t),
    .pNBTxnBuf                = g_MspiDisplayQBuffer,
};

//*****************************************************************************
//
// Main
//
//*****************************************************************************
int
main(void)
{
    uint32_t      ui32Status;
    am_hal_burst_avail_e          eBurstModeAvailable;
    am_hal_burst_mode_e           eBurstMode;

    //
    // Set the clock frequency.
    //
    am_hal_clkgen_control(AM_HAL_CLKGEN_CONTROL_SYSCLK_MAX, 0);

    //
    // Set the default cache configuration
    //
    am_hal_cachectrl_config(&am_hal_cachectrl_defaults);
    am_hal_cachectrl_enable();

    //
    // Configure the board for low power operation.
    //
    am_bsp_low_power_init();

    //
    // Initialize the printf interface for ITM output
    //
    am_bsp_itm_printf_enable();

    //
    // Print the banner.
    //
    am_util_stdio_terminal_clear();
    am_util_stdio_printf("Hello World!\n\n");
    
    // Check that the Burst Feature is available.
    if (AM_HAL_STATUS_SUCCESS == am_hal_burst_mode_initialize(&eBurstModeAvailable))
    {
        if (AM_HAL_BURST_AVAIL == eBurstModeAvailable)
        {
            am_util_stdio_printf("Apollo3 Burst Mode is Available\n");
        }
        else
        {
            am_util_stdio_printf("Apollo3 Burst Mode is Not Available\n");
        }
    }
    else
    {
        am_util_stdio_printf("Failed to Initialize for Burst Mode operation\n");
    }

    // Make sure we are in "Normal" mode.
    if (AM_HAL_STATUS_SUCCESS == am_hal_burst_mode_disable(&eBurstMode))
    {
        if (AM_HAL_NORMAL_MODE == eBurstMode)
        {
            am_util_stdio_printf("Apollo3 operating in Normal Mode (48MHz)\n");
        }
    }
    else
    {
        am_util_stdio_printf("Failed to Disable Burst Mode operation\n");
    }

    // Put the MCU into "Burst" mode.
    if (AM_HAL_STATUS_SUCCESS == am_hal_burst_mode_enable(&eBurstMode))
    {
        if (AM_HAL_BURST_MODE == eBurstMode)
        {
            am_util_stdio_printf("Apollo3 operating in Burst Mode (96MHz)\n");
        }
    }
    else
    {
        am_util_stdio_printf("Failed to Enable Burst Mode operation\n");
    }
 
    am_hal_interrupt_master_enable();


    am_devices_mspi_display_gpio_init();
    ui32Status = am_devices_mspi_rm69330_init(MSPI_TEST_MODULE, (am_devices_mspi_rm69330_config_t *)&QuadDisplayMSPICfg, &g_DisplayHandle, &g_MSPIDisplayHandle);
    
    if (AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS != ui32Status)
    {
        am_util_stdio_printf("Failed to configure the MSPI and Display Device correctly!\n");
    }

    NVIC_EnableIRQ(mspi_psram_interrupts[MSPI_TEST_MODULE]);

    am_devices_mspi_rm69330_display_on(g_DisplayHandle);
    
    AM_HAL_GPIO_MASKCREATE(GpioIntMask_TE);
    am_hal_gpio_interrupt_clear(AM_HAL_GPIO_MASKBIT(pGpioIntMask_TE, AM_BSP_GPIO_DISPLAY_HBOARD_TE));
    am_hal_gpio_interrupt_register(AM_BSP_GPIO_DISPLAY_HBOARD_TE, teInt_handler);
    am_hal_gpio_interrupt_enable(AM_HAL_GPIO_MASKBIT(pGpioIntMask_TE, AM_BSP_GPIO_DISPLAY_HBOARD_TE));
    NVIC_EnableIRQ(GPIO_IRQn);

    mspi_psram_init();

    init_mspi_psram_data();

    uint32_t j = 0;
    while(1)
    {
        #if 1
        //todo: this is for up-down slide
        while (g_TE_complete == 0);
        ui32Status = am_devices_mspi_psram_read(g_PsramHandle, (uint8_t *)&g_FB[0], (340*3*j), FB_SIZE, true);
        if (AM_DEVICES_MSPI_PSRAM_STATUS_SUCCESS != ui32Status)
        {
            am_util_stdio_printf("Failed to read buffer 1 to PSRAM Device!\n");
            return -1;
        }
        
        ui32Status = am_devices_mspi_rm69330_nonblocking_write(g_DisplayHandle, &g_FB[0], FB_SIZE, true);

        j++;
        if (j == 68*5) {
            j = 0;
        }
        g_TE_complete = 0;
        #else

        #endif
    }
}
