//*****************************************************************************
//
//! @file am_devices_mspi_rm69330.c
//!
//! @brief General Multibit SPI Display driver.
//
//*****************************************************************************

//*****************************************************************************
//
// ${copyright}
//
// This is part of revision ${version} of the AmbiqSuite Development Package.
//
//*****************************************************************************
#include "am_mcu_apollo.h"
#include "am_devices_mspi_rm69330.h"
#include "am_util_stdio.h"
#include "am_util_delay.h"

//*****************************************************************************
//
// Global variables.
//
//*****************************************************************************
#define BYTE_NUM_PER_WRITE  65535
#define AM_DEVICES_MSPI_RM69330_TIMEOUT             100000

static struct
{
    uint32_t row_start;
    uint32_t row_end;
    uint32_t col_start;
    uint32_t col_end;
} gs_display_info;

static am_devices_mspi_rm69330_graphic_conf_t g_sGraphic_conf =
{
    .bus_mode       = AM_DEVICES_MSPI_RM69330_SPI_WRAM,
    .color_mode     = AM_DEVICES_MSPI_RM69330_COLOR_MODE_24BIT,
    .scan_mode      = AM_DEVICES_MSPI_RM69330_SCAN_MODE_270,
    .max_row        = 340,
    .max_col        = 340,
    .row_offset     = 10,
    .col_offset     = 10
};

typedef struct
{
    uint32_t                    ui32Module;
    void                        *pMspiHandle;
    am_hal_mspi_device_e        eDeviceConfig;
    bool                        bOccupied;
} am_devices_mspi_rm69330_t;

static am_hal_mspi_dev_config_t  QuadCE0DisplayMSPICfg =
{
    .eSpiMode             = AM_HAL_MSPI_SPI_MODE_0,
    .eClockFreq           = AM_HAL_MSPI_CLK_48MHZ,
    .ui8TurnAround        = 0,
    .eAddrCfg             = AM_HAL_MSPI_ADDR_3_BYTE,
    .eInstrCfg            = AM_HAL_MSPI_INSTR_1_BYTE,
    .eDeviceConfig        = AM_HAL_MSPI_FLASH_QUAD_CE0,
    .bSeparateIO          = false,
    .bSendInstr           = true,
    .bSendAddr            = true,
    .bTurnaround          = false,
    .eXipMixedMode        = AM_HAL_MSPI_XIPMIXED_NORMAL,
#if defined(AM_PART_APOLLO3P)
    .ui8WriteLatency      = 0,
    .bEnWriteLatency      = false,
    .bEmulateDDR          = false,
    .ui16DMATimeLimit     = 0,
    .eDMABoundary         = AM_HAL_MSPI_BOUNDARY_NONE,
#endif
    .ui32TCBSize          = 0,
    .pTCB                 = NULL,
    .ui8ReadInstr         = AM_DEVICES_MSPI_RM69330_FAST_READ,
    .ui8WriteInstr        = AM_DEVICES_MSPI_RM69330_PIXEL_WRITE_ADDR4,
    .scramblingStartAddr  = 0,
    .scramblingEndAddr    = 0,
};

static am_hal_mspi_dev_config_t  QuadCE1DisplayMSPICfg =
{
    .eSpiMode             = AM_HAL_MSPI_SPI_MODE_0,
    .eClockFreq           = AM_HAL_MSPI_CLK_48MHZ,
    .ui8TurnAround        = 0,
    .eAddrCfg             = AM_HAL_MSPI_ADDR_3_BYTE,
    .eInstrCfg            = AM_HAL_MSPI_INSTR_1_BYTE,
    .eDeviceConfig        = AM_HAL_MSPI_FLASH_QUAD_CE1,
    .bSeparateIO          = false,
    .bSendInstr           = true,
    .bSendAddr            = true,
    .bTurnaround          = false,
    .eXipMixedMode        = AM_HAL_MSPI_XIPMIXED_NORMAL,
#if defined(AM_PART_APOLLO3P)
    .ui8WriteLatency      = 0,
    .bEnWriteLatency      = false,
    .bEmulateDDR          = false,
    .ui16DMATimeLimit     = 0,
    .eDMABoundary         = AM_HAL_MSPI_BOUNDARY_NONE,
#endif
    .ui32TCBSize          = 0,
    .pTCB                 = NULL,
    .ui8ReadInstr         = AM_DEVICES_MSPI_RM69330_FAST_READ,
    .ui8WriteInstr        = AM_DEVICES_MSPI_RM69330_PIXEL_WRITE_ADDR4,
    .scramblingStartAddr  = 0,
    .scramblingEndAddr    = 0,
};

am_devices_mspi_rm69330_t gAmDisplay[AM_DEVICES_MSPI_RM69330_MAX_DEVICE_NUM];

void pfnMSPI_RM69330_Callback(void *pCallbackCtxt, uint32_t status)
{
    // Set the DMA complete flag.
    *(volatile bool *)pCallbackCtxt = true;
}

static uint32_t
am_devices_mspi_rm69330_command_write(void *pHandle, uint8_t ui8Instr,
                                      uint8_t *pData,
                                      uint32_t ui32NumBytes)
{
    am_hal_mspi_pio_transfer_t  Transaction;
    am_devices_mspi_rm69330_t *pDisplay = (am_devices_mspi_rm69330_t *)pHandle;
    uint8_t command_buf[8] = {0x0};
    if ( ui32NumBytes > 4 )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR; // too many bytes
    }

    command_buf[0] = AM_DEVICES_MSPI_RM69330_CMD_WRITE;
    command_buf[1] = 0x00;
    command_buf[2] = ui8Instr;
    command_buf[3] = 0x00;

    // build parameter
    if ( ui32NumBytes > 0 )
    {
        for ( uint8_t i = 0; i < ui32NumBytes; i++ )
        {
            command_buf[4 + i] = pData[i];
        }
    }

    // Create the individual write transaction.
    Transaction.ui32NumBytes       = (ui32NumBytes + 4);
    Transaction.bScrambling        = false;
    Transaction.eDirection         = AM_HAL_MSPI_TX;
    Transaction.bSendAddr          = false;
    Transaction.ui32DeviceAddr     = NULL;
    Transaction.bSendInstr         = false;
    Transaction.ui16DeviceInstr    = NULL;
    Transaction.bTurnaround        = false;
    Transaction.bQuadCmd           = false;
#if defined(AM_PART_APOLLO3P)
    Transaction.bEnWRLatency       = false;
    Transaction.bDCX               = false;
    Transaction.bContinue          = false;
#endif
    Transaction.pui32Buffer        = (uint32_t *)command_buf;

    // Execute the transction over MSPI.
    return am_hal_mspi_blocking_transfer(pDisplay->pMspiHandle, &Transaction,
                                         AM_DEVICES_MSPI_RM69330_TIMEOUT);
}

//*****************************************************************************
//
//! @brief Reads the current status of the external display
//!
//! @param ui32DeviceNumber - Device number of the external display
//!
//! This function reads the device ID register of the external display, and returns
//! the result as an 32-bit unsigned integer value.
//!
//! @return 32-bit status
//
//*****************************************************************************
static void
am_devices_mspi_rm69330_reset(void)
{
    am_hal_gpio_output_clear(AM_BSP_GPIO_DISPLAY_RESET);  //AMOLED_RESET
    am_util_delay_ms(20);
    am_hal_gpio_output_set(AM_BSP_GPIO_DISPLAY_RESET);

    am_util_delay_ms(150);  //Delay 150ms
}

static uint32_t
am_devices_set_row_col(void *pHandle, am_devices_mspi_rm69330_graphic_conf_t *psGraphic_conf)
{
    uint8_t data[4] = {0};

    gs_display_info.row_start = psGraphic_conf->row_offset;
    gs_display_info.row_end = psGraphic_conf->max_row + psGraphic_conf->row_offset - 1;
    gs_display_info.col_start = psGraphic_conf->col_offset;
    gs_display_info.col_end = psGraphic_conf->max_col + psGraphic_conf->col_offset - 1;

    /* set column start address */
    data[0] = (gs_display_info.col_start / 256);
    data[1] = (gs_display_info.col_start % 256);
    data[2] = (gs_display_info.col_end / 256);
    data[3] = (gs_display_info.col_end % 256);
    if (am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SET_COLUMN, data, 4))//Column
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    /* set row start address */
    data[0] = (gs_display_info.row_start / 256);
    data[1] = (gs_display_info.row_start % 256);
    data[2] = (gs_display_info.row_end / 256);
    data[3] = (gs_display_info.row_end % 256);
    if (am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SET_ROW, data, 4))//raw
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

static uint32_t
am_devices_lcm_init(void *pHandle, am_devices_mspi_rm69330_graphic_conf_t *psGraphic_conf)
{
    uint8_t cmd_buf[10];

    am_devices_mspi_rm69330_reset();

    cmd_buf[0] = 0x00;      //set to page CMD1
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_CMD_MODE, cmd_buf, 1) )  // set page CMD1
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    cmd_buf[0] = psGraphic_conf->bus_mode;
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SET_DSPI_MODE, cmd_buf, 1) )  // spi ram enable
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_IDLE_MODE_OFF, NULL, 0) )  // idle mode off
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    am_util_delay_ms(120);

    //--------------------------------------Display Setting------------------------------------------//
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_DISPLAY_OFF, NULL, 0) )  // display off
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    cmd_buf[0] = psGraphic_conf->scan_mode;
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SCAN_DIRECTION, cmd_buf, 1) )  // scan direction to 0
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    cmd_buf[0] = psGraphic_conf->color_mode;      // Interface Pixel Format, 0x111 = 24bit (RGB888) ; 0x101 = 16bit (RGB565)
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_PIXEL_FORMAT, cmd_buf, 1) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    cmd_buf[0] = 0x02;      // TE on , only V-blanking
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_TE_LINE_ON, cmd_buf, 1) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    cmd_buf[0] = 0xff;      // write display brightness
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_WRITE_DISPLAY_BRIGHTNESS, cmd_buf, 1) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    am_util_delay_ms(10);

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SLEEP_OUT, NULL, 0) )  // sleep out
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    am_util_delay_ms(10);
    //--------------------------------------Display Setting------------------------------------------//
    am_devices_set_row_col(pHandle, psGraphic_conf);   // upper half of the display

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_NORMAL_MODE_ON, NULL, 0) )  // normal display on
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_IDLE_MODE_OFF, NULL, 0) )  // idle mode off
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SLEEP_OUT, NULL, 0) )  // sleep out
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    am_util_delay_ms(10);
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

//*****************************************************************************
//
//! @brief Initialize the MSPI_RM69330 driver.
//!
//! @param psIOMSettings - IOM device structure describing the target screen.
//! @param pfnWriteFunc - Function to use for spi writes.
//! @param pfnReadFunc - Function to use for spi reads.
//!
//! This function should be called before any other am_devices_MSPI_RM69330
//! functions. It is used to set tell the other functions how to communicate
//! with the external screen hardware.
//!
//! The \e pfnWriteFunc and \e pfnReadFunc variables may be used to provide
//! alternate implementations of SPI write and read functions respectively. If
//! they are left set to 0, the default functions am_hal_iom_spi_write() and
//! am_hal_iom_spi_read() will be used.
//!
//! @return None.
//
//*****************************************************************************
uint32_t
am_devices_mspi_rm69330_init(uint32_t ui32Module, am_devices_mspi_rm69330_config_t *pDevCfg, void **ppHandle, void **ppMspiHandle)
{
    uint32_t      ui32Status;
    uint32_t      ui32Index;
    am_hal_mspi_dev_config_t    mspiDevCfg;
    void                        *pMspiHandle;

    if ((ui32Module > AM_REG_MSPI_NUM_MODULES) || (pDevCfg == NULL))
    {
      return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    //
    // Enable fault detection.
    //
#if AM_APOLLO3_MCUCTRL
    am_hal_mcuctrl_control(AM_HAL_MCUCTRL_CONTROL_FAULT_CAPTURE_ENABLE, 0);
#else // AM_APOLLO3_MCUCTRL
    am_hal_mcuctrl_fault_capture_enable();
#endif // AM_APOLLO3_MCUCTRL

    // Allocate a vacant device handle
    for ( ui32Index = 0; ui32Index < AM_DEVICES_MSPI_RM69330_MAX_DEVICE_NUM; ui32Index++ )
    {
        if ( gAmDisplay[ui32Index].bOccupied == false )
        {
            break;
        }
    }
    if ( ui32Index == AM_DEVICES_MSPI_RM69330_MAX_DEVICE_NUM )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    switch (pDevCfg->eDeviceConfig)
    {
        case AM_HAL_MSPI_FLASH_SERIAL_CE0:
        case AM_HAL_MSPI_FLASH_QUAD_CE0:
            mspiDevCfg = QuadCE0DisplayMSPICfg;
            break;
        case AM_HAL_MSPI_FLASH_SERIAL_CE1:
        case AM_HAL_MSPI_FLASH_QUAD_CE1:
            mspiDevCfg = QuadCE1DisplayMSPICfg;
            break;
        case AM_HAL_MSPI_FLASH_DUAL_CE0:
        case AM_HAL_MSPI_FLASH_OCTAL_CE0:
        case AM_HAL_MSPI_FLASH_DUAL_CE1:
        case AM_HAL_MSPI_FLASH_OCTAL_CE1:
        case AM_HAL_MSPI_FLASH_QUADPAIRED:
        case AM_HAL_MSPI_FLASH_QUADPAIRED_SERIAL:
            return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    mspiDevCfg.eClockFreq = pDevCfg->eClockFreq;
    mspiDevCfg.eXipMixedMode = pDevCfg->eMixedMode;
    mspiDevCfg.ui32TCBSize = pDevCfg->ui32NBTxnBufLength;
    mspiDevCfg.pTCB = pDevCfg->pNBTxnBuf;

    //
    // Configure the MSPI for Serial or Quad-Paired Serial operation during initialization.
    //
    if (AM_HAL_STATUS_SUCCESS != am_hal_mspi_initialize(ui32Module, &pMspiHandle))
    {
        am_util_stdio_printf("Error - Failed to initialize MSPI.\n");
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if (AM_HAL_STATUS_SUCCESS != am_hal_mspi_power_control(pMspiHandle, AM_HAL_SYSCTRL_WAKE, false))
    {
        am_util_stdio_printf("Error - Failed to power on MSPI.\n");
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if (AM_HAL_STATUS_SUCCESS != am_hal_mspi_device_configure(pMspiHandle, &mspiDevCfg))
    {
        am_util_stdio_printf("Error - Failed to configure MSPI.\n");
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    if (AM_HAL_STATUS_SUCCESS != am_hal_mspi_enable(pMspiHandle))
    {
        am_util_stdio_printf("Error - Failed to enable MSPI.\n");
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    am_bsp_mspi_pins_enable(ui32Module, pDevCfg->eDeviceConfig);

    gAmDisplay[ui32Index].pMspiHandle = pMspiHandle;
    gAmDisplay[ui32Index].ui32Module = ui32Module;

    am_devices_lcm_init((void*)&gAmDisplay[ui32Index], &g_sGraphic_conf);

    //
    // Enable MSPI interrupts.
    //
    ui32Status = am_hal_mspi_interrupt_clear(pMspiHandle, AM_HAL_MSPI_INT_CQUPD | AM_HAL_MSPI_INT_ERR );
    if (AM_HAL_STATUS_SUCCESS != ui32Status)
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    ui32Status = am_hal_mspi_interrupt_enable(pMspiHandle, AM_HAL_MSPI_INT_CQUPD | AM_HAL_MSPI_INT_ERR );
    if (AM_HAL_STATUS_SUCCESS != ui32Status)
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    //
    // Return the handle.
    //
    gAmDisplay[ui32Index].bOccupied = true;
    *ppHandle = (void *)&gAmDisplay[ui32Index];
    *ppMspiHandle = pMspiHandle;
    gAmDisplay[ui32Index].eDeviceConfig = mspiDevCfg.eDeviceConfig;

    //
    // Return the status.
    //
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

//*****************************************************************************
//
//! @brief De-Initialize the mspi_rm69330 driver.
//!
//! @param ui32Module     - MSPI Module#
//!
//! This function reverses the initialization
//!
//! @return Status.
//
//*****************************************************************************
uint32_t
am_devices_mspi_rm69330_term(void *pHandle)
{
    am_devices_mspi_rm69330_t *pDisplay = (am_devices_mspi_rm69330_t *)pHandle;
    uint32_t    ui32Status;

    if ( pDisplay->ui32Module > AM_REG_IOM_NUM_MODULES )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    //
    // Disable and clear the interrupts to start with.
    //
    ui32Status = am_hal_mspi_interrupt_disable(pDisplay->pMspiHandle, 0xFFFFFFFF);
    if (AM_HAL_STATUS_SUCCESS != ui32Status)
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    ui32Status = am_hal_mspi_interrupt_clear(pDisplay->pMspiHandle, 0xFFFFFFFF);
    if (AM_HAL_STATUS_SUCCESS != ui32Status)
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    //
    // Disable the MSPI.
    //
    am_hal_mspi_disable(pDisplay->pMspiHandle);

    //
    // Disable power to and uninitialize the MSPI instance.
    //
    am_hal_mspi_power_control(pDisplay->pMspiHandle, AM_HAL_SYSCTRL_DEEPSLEEP, false);

    am_hal_mspi_deinitialize(pDisplay->pMspiHandle);

    // Free this device handle
    pDisplay->bOccupied = false;

    //
    // Return the status.
    //
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}


uint32_t
am_devices_mspi_rm69330_display_off(void *pHandle)
{
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_DISPLAY_OFF, NULL, 0) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SLEEP_IN, NULL, 0) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

uint32_t
am_devices_mspi_rm69330_display_on(void *pHandle)
{
    if (am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_DISPLAY_ON, NULL, 0))
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

//*****************************************************************************
//
//! @brief Programs the given range of display addresses.
//!
//! @param ui32Module - MSPI Instance
//! @param pui8TxBuffer - Buffer to write the data from
//! @param ui32NumBytes - Number of bytes to write to the display memory
//! @param bWaitForCompletion - Waits for CQ/DMA to complete before return.
//!
//! This function uses the data in the provided pui8TxBuffer and copies it to
//! the external flash at the address given by ui32WriteAddress. It will copy
//! exactly ui32NumBytes of data from the original pui8TxBuffer pointer. The
//! user is responsible for ensuring that they do not overflow the target flash
//! memory or underflow the pui8TxBuffer array
//!
//! @return 32-bit status
//
//*****************************************************************************
uint32_t
am_devices_mspi_rm69330_nonblocking_write(void *pHandle, const uint8_t *pui8TxBuffer,
                                          uint32_t ui32NumBytes,
                                          bool bWaitForCompletion)
{
    am_hal_mspi_dma_transfer_t    Transaction;
    am_devices_mspi_rm69330_t *pDisplay = (am_devices_mspi_rm69330_t *)pHandle;
    uint32_t                      ui32Status;
    uint32_t                      ui32BytesLeft = ui32NumBytes;
    volatile bool bDMAComplete = false;

    while (ui32BytesLeft)
    {
        if ( ui32BytesLeft == ui32NumBytes )
        {
            Transaction.ui32DeviceAddress = AM_DEVICES_MSPI_RM69330_MEM_WRITE << 8;
        }
        else
        {
            Transaction.ui32DeviceAddress = AM_DEVICES_MSPI_RM69330_MEM_WRITE_CONTINUE << 8;
        }
        // Set the DMA priority
        Transaction.ui8Priority = 1;

        // Set the transfer direction to TX (Write)
        Transaction.eDirection = AM_HAL_MSPI_TX;

        Transaction.ui32TransferCount = (ui32BytesLeft > BYTE_NUM_PER_WRITE) ? BYTE_NUM_PER_WRITE : ui32BytesLeft;

        // Set the source SRAM buffer address.
        Transaction.ui32SRAMAddress = (uint32_t)pui8TxBuffer;

        // Clear the CQ stimulus.
        Transaction.ui32PauseCondition = 0;
        // Clear the post-processing
        Transaction.ui32StatusSetClr = 0;

        // Start the transaction.
        bDMAComplete = false;
        ui32Status = am_hal_mspi_nonblocking_transfer(pDisplay->pMspiHandle, &Transaction, AM_HAL_MSPI_TRANS_DMA, pfnMSPI_RM69330_Callback, (void *)&bDMAComplete);

        // Check the transaction status.
        if (AM_HAL_STATUS_SUCCESS != ui32Status)
        {
            return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
        }

        if (bWaitForCompletion)
        {
            // Wait for DMA Complete or Timeout
            for (uint32_t i = 0; i < AM_DEVICES_MSPI_RM69330_TIMEOUT; i++)
            {
                if (bDMAComplete)
                {
                    break;
                }
                //
                // Call the BOOTROM cycle function to delay for about 1 microsecond.
                //
                am_hal_flash_delay( FLASH_CYCLES_US(1) );
            }

            // Check the status.
            if (!bDMAComplete)
            {
                return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
            }
        }
        ui32BytesLeft -= Transaction.ui32TransferCount;
        pui8TxBuffer += Transaction.ui32TransferCount;
    }

    //
    // Return the status.
    //
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

//*****************************************************************************
//
//! @brief Programs the given range of display addresses.
//!
//! @param ui32Module - MSPI Instance
//! @param pui8TxBuffer - Buffer to write the data from
//! @param ui32NumBytes - Number of bytes to write to the display memory
//! @param ui32PauseCondition - CQ Pause condition before execution.
//! @param ui32StatusSetClr - CQ Set/Clear condition after execution.
//! @param pfnCallback - Callback function after execution.
//! @param pCallbackCtxt - Callback context after execution.
//!
//! This function uses the data in the provided pui8TxBuffer and copies it to
//! the external display the address given by ui32WriteAddress. It will copy
//! exactly ui32NumBytes of data from the original pui8TxBuffer pointer. The
//! user is responsible for ensuring that they do not overflow the target display
//! memory or underflow the pui8TxBuffer array
//
//! @return 32-bit status
//
//*****************************************************************************
uint32_t
am_devices_mspi_rm69330_nonblocking_write_adv(void *pHandle,
                                              uint8_t *pui8TxBuffer,
                                              uint32_t ui32NumBytes,
                                              uint32_t ui32PauseCondition,
                                              uint32_t ui32StatusSetClr,
                                              am_hal_mspi_callback_t pfnCallback,
                                              void *pCallbackCtxt)
{
    am_hal_mspi_dma_transfer_t Transaction;
    am_devices_mspi_rm69330_t *pDisplay = (am_devices_mspi_rm69330_t *)pHandle;

    //
    // Create the transaction.
    //
    Transaction.ui8Priority               = 1;
    Transaction.eDirection                = AM_HAL_MSPI_TX;
    Transaction.ui32TransferCount         = ui32NumBytes;
    Transaction.ui32DeviceAddress         = AM_DEVICES_MSPI_RM69330_MEM_WRITE << 8;
    Transaction.ui32SRAMAddress           = (uint32_t)pui8TxBuffer;
    Transaction.ui32PauseCondition        = ui32PauseCondition;
    Transaction.ui32StatusSetClr          = ui32StatusSetClr;

    //
    // Execute the transction over MSPI.
    //
    if (am_hal_mspi_nonblocking_transfer(pDisplay->pMspiHandle,
                                         &Transaction,
                                         AM_HAL_MSPI_TRANS_DMA,
                                         pfnCallback,
                                         pCallbackCtxt))
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

uint32_t
am_devices_mspi_rm69330_row_col_reset(void *pHandle)
{
    uint32_t ui32Status = am_devices_set_row_col(pHandle, &g_sGraphic_conf);
    if (AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS != ui32Status)
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

uint32_t am_devices_mspi_rm69330_set_transfer_window(void *pHandle, uint16_t col_start, uint16_t col_end,
                                                     uint16_t row_start, uint16_t row_end)
{
    uint8_t cmd_buf[4];
    cmd_buf[0] = (col_start / 256);
    cmd_buf[1] = (col_start % 256);
    cmd_buf[2] = (col_end / 256);
    cmd_buf[3] = (col_end % 256);
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SET_COLUMN, cmd_buf, 4) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }

    am_util_delay_us(10);

    cmd_buf[0] = (row_start / 256);
    cmd_buf[1] = (row_start % 256);
    cmd_buf[2] = (row_end / 256);
    cmd_buf[3] = (row_end % 256);
    if ( am_devices_mspi_rm69330_command_write(pHandle, AM_DEVICES_MSPI_RM69330_SET_ROW, cmd_buf, 4) )
    {
        return AM_DEVICES_MSPI_RM69330_STATUS_ERROR;
    }
    return AM_DEVICES_MSPI_RM69330_STATUS_SUCCESS;
}

